<?php
require_once '../config/cors.php';
require_once '../config/database.php';
require_once '../middleware/auth.php';
require_once '../utils/jwt.php';

// --- DATABASE CONNECTION ---
$database = new Database();
$db = $database->getConnection();

// --- HTTP METHOD HANDLING ---
$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        try {
            $stmt = $db->prepare("
                SELECT n.*, u.name as user_name, u.email 
                FROM notifications n 
                LEFT JOIN users u ON n.user_id = u.id 
                ORDER BY n.created_at DESC 
                LIMIT 100
            ");
            $stmt->execute();
            $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Add a recipient_type for frontend logic
            foreach ($notifications as &$notification) {
                if (is_null($notification['user_id'])) {
                    $notification['recipient_type'] = 'all';
                    $notification['user_name'] = 'All Users';
                }
            }
            
            http_response_code(200);
            echo json_encode(['success' => true, 'notifications' => $notifications]);

        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
        }
        break;

    case 'POST':
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['title']) || !isset($data['message']) || !isset($data['type'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Missing required fields: title, message, and type']);
            exit;
        }

        $db->beginTransaction();
        try {
            // --- THIS IS THE FIX FOR "SEND TO ALL" ---
            if (isset($data['recipient_type']) && $data['recipient_type'] === 'all') {
                // Get all user IDs
                $userStmt = $db->prepare("SELECT id FROM users");
                $userStmt->execute();
                $users = $userStmt->fetchAll(PDO::FETCH_ASSOC);

                // Prepare statement for inserting notifications
                $stmt = $db->prepare("INSERT INTO notifications (user_id, title, message, type) VALUES (:user_id, :title, :message, :type)");

                foreach ($users as $user) {
                    $stmt->execute([
                        ':user_id' => $user['id'],
                        ':title' => $data['title'],
                        ':message' => $data['message'],
                        ':type' => $data['type']
                    ]);
                }
                $message = 'Notification sent to all users successfully';

            } else {
                // --- SEND TO A SINGLE USER ---
                if (!isset($data['user_id'])) {
                    http_response_code(400);
                    echo json_encode(['success' => false, 'error' => 'User ID is required for individual notifications']);
                    exit;
                }
                
                $stmt = $db->prepare("INSERT INTO notifications (user_id, title, message, type) VALUES (:user_id, :title, :message, :type)");
                $stmt->execute([
                    ':user_id' => $data['user_id'],
                    ':title' => $data['title'],
                    ':message' => $data['message'],
                    ':type' => $data['type']
                ]);
                $message = 'Notification created successfully';
            }
            
            $db->commit();
            http_response_code(201);
            echo json_encode(['success' => true, 'message' => $message]);

        } catch (PDOException $e) {
            $db->rollBack();
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Failed to create notification: ' . $e->getMessage()]);
        }
        break;

    case 'PUT':
        // --- ADDED: FULL EDIT FUNCTIONALITY ---
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Notification ID is required']);
            exit;
        }
        
        try {
            $updates = [];
            $params = [':id' => $data['id']];

            // Whitelist fields that can be updated
            if (isset($data['title'])) { $updates[] = "title = :title"; $params[':title'] = $data['title']; }
            if (isset($data['message'])) { $updates[] = "message = :message"; $params[':message'] = $data['message']; }
            if (isset($data['read_status'])) { $updates[] = "read_status = :read_status"; $params[':read_status'] = (int)$data['read_status']; }

            if (empty($updates)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'No fields to update']);
                exit;
            }
            
            $sql = "UPDATE notifications SET " . implode(", ", $updates) . " WHERE id = :id";
            $stmt = $db->prepare($sql);
            
            if ($stmt->execute($params)) {
                 http_response_code(200);
                 echo json_encode(['success' => true, 'message' => 'Notification updated successfully']);
            } else {
                http_response_code(500);
                echo json_encode(['success' => false, 'error' => 'Failed to execute update statement']);
            }

        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Failed to update notification: ' . $e->getMessage()]);
        }
        break;

    case 'DELETE':
        // --- ADDED: DELETE FUNCTIONALITY ---
        $id = $_GET['id'] ?? null;
        
        if (!$id) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Notification ID is required in the URL.']);
            exit;
        }
        
        try {
            $stmt = $db->prepare("DELETE FROM notifications WHERE id = :id");
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                http_response_code(200);
                echo json_encode(['success' => true, 'message' => 'Notification deleted successfully']);
            } else {
                http_response_code(500);
                echo json_encode(['success' => false, 'error' => 'Failed to delete notification']);
            }
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Database error on delete: ' . $e->getMessage()]);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'Method not allowed']);
        break;
}
?>