<?php
/**
 * Product Management API
 *
 * Handles CRUD operations for products, including related images and variants.
 * Uses PDO for secure database interactions and transactions for data integrity.
 */

// --- BOOTSTRAPPING & CONFIGURATION ---
ini_set('display_errors', 1);
error_reporting(E_ALL);
header("Content-Type: application/json; charset=UTF-8");

require_once '../config/cors.php';
require_once '../config/database.php';

// --- CONSTANTS ---
define('PRODUCT_UPLOAD_DIR', __DIR__ . '/../uploads/products/');
define('MAX_FILE_SIZE', 50 * 1024 * 1024); // 50 MB
define('MAX_IMAGE_COUNT', 10);
$allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];

// --- DATABASE CONNECTION ---
try {
    $database = new Database();
    $db = $database->getConnection();
} catch (Exception $e) {
    http_response_code(503);
    echo json_encode(['error' => 'Service unavailable. Could not connect to the database.']);
    exit();
}

// --- HELPER FUNCTIONS ---

/**
 * Sends a standardized JSON response and sets the HTTP status code.
 */
function sendJsonResponse($statusCode, $data) {
    http_response_code($statusCode);
    echo json_encode($data);
}

/**
 * Generates a standard v4 UUID.
 */
function generateUuidV4() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

/**
 * Constructs the full public URL for a stored image filename.
 */
function getImageUrl($imageName) {
    if (empty($imageName)) {
        return null;
    }
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https://" : "http://";
    $host = $_SERVER['HTTP_HOST'];
    // Correctly resolve the base path for the uploads directory
    $basePath = str_replace($_SERVER['DOCUMENT_ROOT'], '', realpath(PRODUCT_UPLOAD_DIR));
    return $protocol . $host . $basePath . '/' . $imageName;
}

/**
 * Validates an array of uploaded files against predefined rules.
 * @throws Exception if validation fails.
 */
function validateUploadedFiles($files, $maxCount, $maxSize, $allowedMimes) {
    if (empty($files) || (isset($files['error'][0]) && $files['error'][0] === UPLOAD_ERR_NO_FILE)) {
        return; // No files to validate is a valid scenario.
    }

    if (count($files['name']) > $maxCount) {
        throw new Exception("You can upload a maximum of {$maxCount} images.", 400);
    }

    foreach ($files['error'] as $key => $error) {
        if ($error !== UPLOAD_ERR_OK) {
            throw new Exception("Error uploading file '{$files['name'][$key]}'. Code: {$error}", 400);
        }
        if ($files['size'][$key] > $maxSize) {
            throw new Exception("File '{$files['name'][$key]}' is too large. Max size is 50MB.", 400);
        }
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime = finfo_file($finfo, $files['tmp_name'][$key]);
        finfo_close($finfo);
        if (!in_array($mime, $allowedMimes)) {
            throw new Exception("Invalid file type for '{$files['name'][$key]}'.", 400);
        }
    }
}

/**
 * Deletes all physical image files and database records for a given product.
 */
function deleteProductImages($db, $productId) {
    $stmt = $db->prepare("SELECT url FROM product_images WHERE product_id = :product_id");
    $stmt->execute([':product_id' => $productId]);
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $filePath = PRODUCT_UPLOAD_DIR . $row['url'];
        if (file_exists($filePath)) {
            unlink($filePath);
        }
    }
    $deleteStmt = $db->prepare("DELETE FROM product_images WHERE product_id = :product_id");
    $deleteStmt->execute([':product_id' => $productId]);
}

/**
 * Inserts new product images into the database and moves uploaded files.
 */
function insertProductImages($db, $productId, $files) {
    if (empty($files) || (isset($files['error'][0]) && $files['error'][0] === UPLOAD_ERR_NO_FILE)) {
        return;
    }
    $stmt = $db->prepare("INSERT INTO product_images (product_id, url, position) VALUES (:product_id, :url, :position)");
    foreach ($files['name'] as $index => $name) {
        $tmp_name = $files['tmp_name'][$index];
        $extension = pathinfo($name, PATHINFO_EXTENSION);
        $newFileName = uniqid('prod_' . substr($productId, 0, 8) . '_', true) . '.' . strtolower($extension);
        $destination = PRODUCT_UPLOAD_DIR . $newFileName;
        
        if (move_uploaded_file($tmp_name, $destination)) {
            $stmt->execute([
                ':product_id' => $productId,
                ':url' => $newFileName,
                ':position' => $index
            ]);
        }
    }
}

/**
 * Replaces product variants: deletes existing ones and inserts new ones.
 */
function handleProductVariants($db, $productId, $variants, $baseSku) {
    // First, remove all existing variants for this product
    $deleteStmt = $db->prepare("DELETE FROM product_variants WHERE product_id = :product_id");
    $deleteStmt->execute([':product_id' => $productId]);

    // If new variants are provided, insert them
    if (!empty($variants) && is_array($variants)) {
        $variantStmt = $db->prepare(
            "INSERT INTO product_variants (product_id, variant_name, price_diff, stock, sku) 
             VALUES (:product_id, :variant_name, :price_diff, :stock, :sku)"
        );
        foreach ($variants as $variant) {
            // Generate a more predictable variant SKU
            $variantSuffix = strtoupper(substr(preg_replace('/[^a-zA-Z0-9]/', '', $variant['variant_name']), 0, 4));
            $variantSku = "{$baseSku}-{$variantSuffix}";
            
            $variantStmt->execute([
                ':product_id' => $productId,
                ':variant_name' => $variant['variant_name'],
                ':price_diff' => $variant['price_diff'] ?? 0,
                ':stock' => $variant['stock'] ?? 0,
                ':sku' => $variantSku
            ]);
        }
    }
}

// --- ENSURE UPLOAD DIRECTORY EXISTS ---
if (!is_dir(PRODUCT_UPLOAD_DIR)) {
    if (!mkdir(PRODUCT_UPLOAD_DIR, 0775, true)) {
        sendJsonResponse(500, ['error' => 'Failed to create upload directory.']);
        exit();
    }
}

// --- API ROUTING ---
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest($db);
            break;
        case 'POST':
            handlePostRequest($db);
            break;
        case 'DELETE':
            handleDeleteRequest($db);
            break;
        default:
            sendJsonResponse(405, ['error' => 'Method not allowed.']);
            break;
    }
} catch (Exception $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    $errorCode = in_array($e->getCode(), [400, 404, 405]) ? $e->getCode() : 500;
    sendJsonResponse($errorCode, ['error' => $e->getMessage()]);
}

/**
 * Handles READ operations for one or all products.
 */
function handleGetRequest($db) {
    // CORRECTED LINE: Replaced deprecated FILTER_SANITIZE_STRING
    $id = filter_input(INPUT_GET, 'id', FILTER_DEFAULT); 

    if ($id) {
        // Fetch a single, detailed product
        $stmt = $db->prepare("SELECT p.*, c.name as category_name FROM products p LEFT JOIN categories c ON p.category_id = c.id WHERE p.id = :id");
        $stmt->execute([':id' => $id]);
        $product = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$product) { throw new Exception("Product not found.", 404); }

        $imagesStmt = $db->prepare("SELECT id, url FROM product_images WHERE product_id = :id ORDER BY position ASC");
        $imagesStmt->execute([':id' => $id]);
        $images = $imagesStmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($images as &$image) { $image['url'] = getImageUrl($image['url']); }
        $product['images'] = $images;

        $variantsStmt = $db->prepare("SELECT * FROM product_variants WHERE product_id = :id");
        $variantsStmt->execute([':id' => $id]);
        $product['variants'] = $variantsStmt->fetchAll(PDO::FETCH_ASSOC);

        sendJsonResponse(200, $product);
    } else {
        // Fetch a list of all products (simplified view)
        $sql = "SELECT p.id, p.name, p.brand, p.base_price, p.stock, p.is_active, c.name as category_name, 
                       (SELECT url FROM product_images WHERE product_id = p.id ORDER BY position ASC LIMIT 1) as main_image 
                FROM products p 
                LEFT JOIN categories c ON p.category_id = c.id 
                ORDER BY p.created_at DESC";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $products = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($products as &$product) {
            $product['main_image'] = getImageUrl($product['main_image']);
        }
        sendJsonResponse(200, ['products' => $products]);
    }
}

/**
 * Handles CREATE and UPDATE operations.
 */
function handlePostRequest($db) {
    if (!isset($_POST['product_data'])) { throw new Exception('Missing product_data field.', 400); }
    $data = json_decode($_POST['product_data'], true);
    if (json_last_error() !== JSON_ERROR_NONE) { throw new Exception('Invalid JSON in product_data field.', 400); }

    $files = $_FILES['images'] ?? null;
    validateUploadedFiles($files, MAX_IMAGE_COUNT, MAX_FILE_SIZE, $GLOBALS['allowedMimeTypes']);

    $db->beginTransaction();

    if (isset($data['id']) && !empty($data['id'])) {
        // --- UPDATE EXISTING PRODUCT ---
        $productId = $data['id'];
        
        $checkStmt = $db->prepare("SELECT sku FROM products WHERE id = :id");
        $checkStmt->execute([':id' => $productId]);
        $sku = $checkStmt->fetchColumn();
        if (!$sku) { throw new Exception("Product with ID {$productId} not found.", 404); }

        $updates = [];
        $params = [':id' => $productId];
        $allowedFields = ['name', 'brand', 'description', 'base_price', 'stock', 'category_id', 'is_active'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updates[] = "{$field} = :{$field}";
                $params[":{$field}"] = ($field === 'is_active') ? (int)$data[$field] : $data[$field];
            }
        }

        if (!empty($updates)) {
            $sql = "UPDATE products SET " . implode(", ", $updates) . " WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
        }

        if (!empty($files) && (!isset($files['error'][0]) || $files['error'][0] !== UPLOAD_ERR_NO_FILE)) {
            deleteProductImages($db, $productId);
            insertProductImages($db, $productId, $files);
        }

        if (isset($data['variants'])) {
            handleProductVariants($db, $productId, $data['variants'], $sku);
        }

        $db->commit();
        sendJsonResponse(200, ['message' => 'Product updated successfully.']);

    } else {
        // --- CREATE NEW PRODUCT ---
        if (!isset($data['name'], $data['base_price'], $data['category_id'], $data['brand'])) {
            throw new Exception('Missing required fields for new product: name, brand, base_price, category_id.', 400);
        }

        $productId = generateUuidV4();
        $sku = strtoupper(substr(preg_replace('/[^a-zA-Z0-9]/', '', $data['name']), 0, 4) . uniqid());
        
        $sql = "INSERT INTO products (id, name, brand, description, base_price, sku, stock, category_id, is_active) 
                VALUES (:id, :name, :brand, :description, :base_price, :sku, :stock, :category_id, :is_active)";
        $stmt = $db->prepare($sql);
        $success = $stmt->execute([
            ':id' => $productId,
            ':name' => $data['name'],
            ':brand' => $data['brand'],
            ':description' => $data['description'] ?? null,
            ':base_price' => $data['base_price'],
            ':sku' => $sku,
            ':stock' => $data['stock'] ?? 0,
            ':category_id' => $data['category_id'],
            ':is_active' => isset($data['is_active']) ? (int)$data['is_active'] : 1
        ]);
        
        if (!$success || $stmt->rowCount() === 0) {
            throw new Exception("Failed to create the product. Please check if the category_id is valid.", 500);
        }

        insertProductImages($db, $productId, $files);
        handleProductVariants($db, $productId, $data['variants'] ?? [], $sku);

        $db->commit();
        sendJsonResponse(201, ['message' => 'Product created successfully', 'product_id' => $productId]);
    }
}

/**
 * Handles DELETE operation for a product.
 */
function handleDeleteRequest($db) {
    // CORRECTED LINE: Replaced deprecated FILTER_SANITIZE_STRING
    $id = filter_input(INPUT_GET, 'id', FILTER_DEFAULT); 
    if (!$id) { throw new Exception('Product ID is required in the URL (e.g., ?id=123).', 400); }

    $db->beginTransaction();

    $checkStmt = $db->prepare("SELECT id FROM products WHERE id = :id");
    $checkStmt->execute([':id' => $id]);
    if ($checkStmt->rowCount() === 0) { throw new Exception("Product not found.", 404); }

    deleteProductImages($db, $id);
    $db->prepare("DELETE FROM product_variants WHERE product_id = :id")->execute([':id' => $id]);
    
    $stmt = $db->prepare("DELETE FROM products WHERE id = :id");
    $stmt->execute([':id' => $id]);
    
    $db->commit();
    sendJsonResponse(200, ['message' => 'Product deleted successfully.']);
}
?>